# final_interactive_studio.py

# بتاريخ 09-10-2025: نستورد المكتبات الأساسية
import streamlit as st
import cv2
import mediapipe as mp
import math
from streamlit_webrtc import webrtc_streamer, VideoTransformerBase

# بتاريخ 09-10-2025: نجهز أدوات ميديا بايب
mp_drawing = mp.solutions.drawing_utils
mp_drawing_styles = mp.solutions.drawing_styles
mp_hands = mp.solutions.hands
mp_face_mesh = mp.solutions.face_mesh

# ----------------------------------------------------------------------------------------------------

# بتاريخ 09-10-2025: دالة مساعدة لحساب المسافة
def calculate_distance(p1, p2):
    return math.sqrt(((p1.x - p2.x) ** 2) + ((p1.y - p2.y) ** 2))

# بتاريخ 09-10-2025: الدالة الذكية لتمييز الأرقام (باللغة الإنجليزية)
def recognize_gestures_english(hand_landmarks):
    landmarks = hand_landmarks.landmark
    fingers_extended = [
        (landmarks[mp_hands.HandLandmark.THUMB_TIP].x < landmarks[mp_hands.HandLandmark.THUMB_IP].x),
        (landmarks[mp_hands.HandLandmark.INDEX_FINGER_TIP].y < landmarks[mp_hands.HandLandmark.INDEX_FINGER_PIP].y),
        (landmarks[mp_hands.HandLandmark.MIDDLE_FINGER_TIP].y < landmarks[mp_hands.HandLandmark.MIDDLE_FINGER_PIP].y),
        (landmarks[mp_hands.HandLandmark.RING_FINGER_TIP].y < landmarks[mp_hands.HandLandmark.RING_FINGER_PIP].y),
        (landmarks[mp_hands.HandLandmark.PINKY_TIP].y < landmarks[mp_hands.HandLandmark.PINKY_PIP].y)
    ]
    num_fingers_extended = fingers_extended.count(True)

    # المنطق نفسه لكن النتائج بالانجليزي
    if not fingers_extended[0] and num_fingers_extended == 1 and fingers_extended[1]: return "Number 1"
    if not fingers_extended[0] and num_fingers_extended == 2 and fingers_extended[1] and fingers_extended[2]: return "Number 2"
    if not fingers_extended[0] and num_fingers_extended == 3 and fingers_extended[1] and fingers_extended[2] and fingers_extended[3]: return "Number 3"
    if not fingers_extended[0] and num_fingers_extended == 4: return "Number 4"
    if num_fingers_extended == 5: return "Number 5 / Stop"
    
    thumb_tip = landmarks[mp_hands.HandLandmark.THUMB_TIP]
    touch_threshold = 0.07
    if calculate_distance(thumb_tip, landmarks[mp_hands.HandLandmark.PINKY_TIP]) < touch_threshold: return "Number 6"
    if calculate_distance(thumb_tip, landmarks[mp_hands.HandLandmark.RING_FINGER_TIP]) < touch_threshold: return "Number 7"
    if calculate_distance(thumb_tip, landmarks[mp_hands.HandLandmark.MIDDLE_FINGER_TIP]) < touch_threshold: return "Number 8"
    if calculate_distance(thumb_tip, landmarks[mp_hands.HandLandmark.INDEX_FINGER_TIP]) < touch_threshold: return "Number 9"
    
    is_thumb_up = fingers_extended[0] and num_fingers_extended == 1
    is_thumb_down = not fingers_extended[0] and num_fingers_extended == 0 and landmarks[mp_hands.HandLandmark.THUMB_TIP].y > landmarks[mp_hands.HandLandmark.THUMB_IP].y

    if is_thumb_up: return "Number 10 / Like"
    if is_thumb_down: return "Dislike"
    
    return ""

# ----------------------------------------------------------------------------------------------------

# الكلاس المسؤول عن معالجة الفيديو
class StudioTransformer(VideoTransformerBase):
    def __init__(self, mode):
        self.mode = mode
        self.hands = mp_hands.Hands(model_complexity=1, max_num_hands=1, min_detection_confidence=0.7, min_tracking_confidence=0.7)
        self.face_mesh = mp_face_mesh.FaceMesh(max_num_faces=1, min_detection_confidence=0.5, min_tracking_confidence=0.5)

    def transform(self, frame):
        image = frame.to_ndarray(format="bgr24")
        image = cv2.flip(image, 1)
        rgb_image = cv2.cvtColor(image, cv2.COLOR_BGR2RGB)
        
        # --- منطق التشغيل حسب الوضع المختار ---
        if 'Hand' in self.mode:
            results = self.hands.process(rgb_image)
            if results.multi_hand_landmarks:
                # التأكد من أن اليد المكتشفة هي اليد المطلوبة
                hand_label = results.multi_handedness[0].classification[0].label
                if (self.mode == 'Right Hand 🤚' and hand_label == 'Right') or \
                   (self.mode == 'Left Hand ✋' and hand_label == 'Left'):
                    
                    hand_landmarks = results.multi_hand_landmarks[0]
                    mp_drawing.draw_landmarks(image, hand_landmarks, mp_hands.HAND_CONNECTIONS, mp_drawing_styles.get_default_hand_landmarks_style(), mp_drawing_styles.get_default_hand_connections_style())
                    gesture = recognize_gestures_english(hand_landmarks)
                    if gesture:
                        wrist_coords = hand_landmarks.landmark[mp_hands.HandLandmark.WRIST]
                        cv2.putText(image, gesture, (int(wrist_coords.x * image.shape[1] - 70), int(wrist_coords.y * image.shape[0] - 20)), cv2.FONT_HERSHEY_SIMPLEX, 1, (255, 255, 255), 2, cv2.LINE_AA)

        elif self.mode == 'Face Mesh (Blink Detection) 👀':
            results = self.face_mesh.process(rgb_image)
            if results.multi_face_landmarks:
                face_landmarks = results.multi_face_landmarks[0]
                mp_drawing.draw_landmarks(image, face_landmarks, mp_face_mesh.FACEMESH_TESSELATION, None, mp_drawing_styles.get_default_face_mesh_tesselation_style())
                
                # --- منطق تمييز الرمش ---
                # نقاط العين اليسرى واليمنى (فوق وتحت)
                left_eye_top = face_landmarks.landmark[159]
                left_eye_bottom = face_landmarks.landmark[145]
                right_eye_top = face_landmarks.landmark[386]
                right_eye_bottom = face_landmarks.landmark[374]
                
                # حساب المسافة العمودية بين الجفون
                left_eye_dist = calculate_distance(left_eye_top, left_eye_bottom)
                right_eye_dist = calculate_distance(right_eye_top, right_eye_bottom)
                
                # إذا كانت المسافة صغيرة جدًا، معناها العين مغمضة
                BLINK_THRESHOLD = 0.015
                if left_eye_dist < BLINK_THRESHOLD and right_eye_dist < BLINK_THRESHOLD:
                    cv2.putText(image, "Blinking!", (50, 50), cv2.FONT_HERSHEY_SIMPLEX, 1.5, (0, 255, 0), 3)

        return image

# ----------------------------------------------------------------------------------------------------

# --- تصميم واجهة التطبيق ---
st.set_page_config(page_title="استوديو IQ.SA التفاعلي", page_icon="🤟", layout="wide")

# --- الشريط الجانبي ---
with st.sidebar:
    st.title("IQ.SA 🤖")
    st.markdown("---")
    st.header("⚙️ اختر الوضع")
    mode = st.selectbox("وش ودك تشغل؟", ('Right Hand 🤚', 'Left Hand ✋', 'Face Mesh (Blink Detection) 👀'))
    st.markdown("---")
    st.header("🧑🏻‍💻 عني أنا، المطور")
    st.markdown("**MTMA حياكم الله أنا محمد ")
    st.markdown("---")
    st.subheader(":تواصل معي")
    st.markdown("""
    - **📧 الإيميل:** [Mtma.1@hotmail.com](mailto:Mtma.1@hotmail.com)
    - **  GitHub: ** [mtma1](https://github.com/mtma1)
    - ** 🔗 LinkedIn:** [Mohammed Alanazi](https://www.linkedin.com/in/mohammed-alanazi-a1014534b/)
                
    - **ْ X ** [@1Mtma] (https://x.com/1Mtma)
    """)
    st.markdown("---")
    st.write("© 2025 IQ.SA. كل الحقوق محفوظة")

# --- المحتوى الرئيسي ---
st.title("🤟 استوديو IQ.SA التفاعلي")
st.write("اختر وضع التشغيل من القائمة الجانبية لتجربة تحليل الحركة مباشرة من الكاميرا")
st.markdown("---")

col1, col2 = st.columns([2, 1])

with col1:
    webrtc_streamer(
        key="studio",
        video_processor_factory=lambda: StudioTransformer(mode),
        media_stream_constraints={"video": True, "audio": False},
        async_processing=True
    )

with col2:
    st.subheader("💡 التعليمات")
    if 'Hand' in mode:
        st.info("ارفع يدك المختارة أمام الكاميرا وجرب تسوي الأرقام من 1 إلى 10 أو علامة اللايك والديسلايك")
    else:
        st.info("ضع وجهك أمام الكاميرا بشكل واضح وجرب ترمش بعينيك")
    
    st.warning("🔒 **تطمن خصوصيتك بأمان.** كل الشغل يصير على جهازك مباشرة. حنا ما نسجل أو نرسل أي بيانات برا")

# --- قسم المطورين ---
st.markdown("---")
with st.expander("👨‍💻 للمطورين: كيف تطور هذا المشروع؟"):
    st.markdown("""
    **حياك الله يا زميلي المطور!** هذا المشروع مبني على أدوات قوية ومفتوحة المصدر، وتقدر تبني فوقه أشياء رهيبة.
    
    - **التقنيات المستخدمة:**
        - **Streamlit:** لبناء واجهة الويب التفاعلية هذي بكل سهولة وبسرعة
        - **MediaPipe:** من جوجل، وهو المحرك الأساسي اللي يتعرف على نقاط اليد والوجه بدقة عالية
        - **OpenCV:** لمعالجة الفيديو والصور اللي تجي من الكاميرا
    
    - **أفكار للتطوير:**
        1.  **إضافة حركات جديدة:** تقدر تعدل على دالة `recognize_gestures_english` وتضيف حركات أكثر (مثل حروف الأبجدية)
        2.  **التحكم في الأجهزة:** ممكن تستخدم حركات اليد للتحكم في الكمبيوتر (مثلًا رفع وخفض الصوت، أو تحريك الماوس)
        3.  **بناء نموذج تعلم آلة حقيقي:** زي ما كنا بنسوي أول، تقدر تجمع بيانات وتدرب نموذج ذكاء اصطناعي يتعرف على مئات الإشارات بدقة، وهذي هي الخطوة الاحترافية
    
    **الكود متاح لك عشان تتعلم منه وتستخدم أي جزء منه في مشاريعك الخاصة. بالتوفيق!**
    """)